<?php

namespace App\Filament\Resources;

// === KUMPULAN SEMUA CLASS YANG KITA BUTUHKAN ===
use App\Filament\Resources\SiteResource\Pages;
use App\Models\PostHistory;
use App\Models\Site;
use App\Services\WordPressPublisher;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class SiteResource extends Resource
{
    protected static ?string $model = Site::class;

    protected static ?string $navigationIcon = 'heroicon-o-globe-alt';

    // Kita biarkan label default "Sites" sesuai keputusan Anda sebelumnya

    /**
     * Mendefinisikan field form untuk membuat dan mengedit data Situs.
     */
    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('name')
                    ->label('Nama Situs')
                    ->required()
                    ->maxLength(255),
                Forms\Components\TextInput::make('url')
                    ->label('URL Situs')
                    ->required()
                    ->url()
                    ->placeholder('https://www.domain.com')
                    ->maxLength(255),
                Forms\Components\TextInput::make('user')
                    ->label('Username')
                    ->required()
                    ->maxLength(255),
                Forms\Components\TextInput::make('pass')
                    ->label('Application Password')
                    ->password() // Membuat input menjadi tipe password (menyembunyikan teks)
                    ->revealable() // Menambahkan ikon mata untuk melihat password
                    ->required(fn (string $context): bool => $context === 'create') // Wajib diisi hanya saat membuat baru
                    ->dehydrated(fn ($state) => filled($state)), // Hanya update password di DB jika field ini diisi saat edit
            ]);
    }

    /**
     * Mendefinisikan kolom-kolom tabel untuk menampilkan daftar Situs.
     */
    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->label('Nama Situs')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('url')
                    ->label('URL'),
                Tables\Columns\TextColumn::make('user')
                    ->label('Username'),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Tanggal Dibuat')
                    ->dateTime('d M Y')
                    ->sortable(),
            ])
            ->actions([
                // === AKSI POSTING KUSTOM KITA DIMULAI DI SINI ===
                Tables\Actions\Action::make('post')
                    ->label('Posting')
                    ->icon('heroicon-o-paper-airplane')
                    ->color('success')
                    ->form([
                        Forms\Components\TextInput::make('title')
                            ->label('Judul Postingan')
                            ->required(),
                        Forms\Components\RichEditor::make('content')
                            ->label('Konten')
                            ->required(),
                    ])
                    ->action(function (array $data, Site $record) {
                        $publisher = new WordPressPublisher();
                        $result = $publisher->publish($record, $data);

                        if ($result) {
                            // Jika berhasil, catat ke history
                            PostHistory::create([
                                'site_id' => $record->id,
                                'post_title' => $data['title'],
                                'post_url' => $result['link'], // Ambil link dari respons API
                                'status' => 'Berhasil',
                            ]);

                            // Tampilkan notifikasi sukses
                            Notification::make()
                                ->title('Postingan berhasil!')
                                ->body('"' . $data['title'] . '" dipublikasikan ke ' . $record->name)
                                ->success()->send();
                        } else {
                            // Jika gagal, catat juga ke history
                             PostHistory::create([
                                'site_id' => $record->id,
                                'post_title' => $data['title'],
                                'post_url' => '#',
                                'status' => 'Gagal',
                            ]);

                            // Tampilkan notifikasi gagal
                            Notification::make()
                                ->title('Gagal memposting')
                                ->body('Periksa log error atau koneksi ke ' . $record->name)
                                ->danger()->send();
                        }
                    }),
                // === AKSI POSTING KUSTOM KITA BERAKHIR DI SINI ===
                
                // Aksi-aksi standar Filament
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
    
    public static function getRelations(): array
    {
        return [
            //
        ];
    }
    
    public static function getPages(): array
    {
        return [
            'index' => Pages\ListSites::route('/'),
            'create' => Pages\CreateSite::route('/create'),
            'edit' => Pages\EditSite::route('/{record}/edit'),
        ];
    }    
}