<?php

namespace App\Services;

use App\Models\Site;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class WordPressPublisher
{
    /**
     * Mempublikasikan sebuah artikel ke situs WordPress target.
     *
     * @param Site $site Objek situs dari database kita.
     * @param array $postData Data postingan ['title', 'content'].
     * @return array|null Mengembalikan data respons dari WordPress jika berhasil, null jika gagal.
     */
    public function publish(Site $site, array $postData): ?array
    {
        // Menggunakan HTTP Client bawaan Laravel, lebih modern dari cURL
        $response = Http::withoutVerifying() 
            ->withBasicAuth($site->user, $site->pass)
            ->timeout(30) // Set timeout 30 detik       
            ->post(rtrim($site->url, '/') . '/wp-json/wp/v2/posts', [
                'title'   => $postData['title'],
                'content' => $postData['content'],
                'status'  => 'publish',
            ]);

        // Cek jika request berhasil (kode status 2xx) dan spesifik 201 (Created)
        if ($response->successful() && $response->status() == 201) {
            Log::info('Berhasil posting ke ' . $site->name);
            return $response->json(); // Kembalikan data post (termasuk link, id, dll)
        } else {
            // Mencatat error ke file log Laravel untuk debugging
            Log::error('Gagal posting ke ' . $site->name, [
                'status' => $response->status(),
                'body' => $response->body(),
            ]);
            return null;
        }
    }
}